﻿using System;
using System.Collections.Generic;
using System.Text;
using IndianHealthService.BMXNet.Services;

namespace IndianHealthService.BMXNet
{

    /// <summary>
    /// <para>
    /// Each BMX connection to RPMS contains a single RemoteSessionPool with at least
    /// one Session, the primary session.  Applications that need additional server processes
    /// beyond what can be done with async commands can used the RemoteSessionPool.
    /// </para>
    /// <para>
    /// Access to the RemoteSessionPool is accomplished by implementing the RemoteSessionPoolConsumer
    /// interface on your component/control.  Secondary sessions can be opened and closed as they
    /// are needed.  If the AvailableSessionCount is zero then a null RemoteSession is returned so it
    /// is recommend to first check the if a pool HasAvailableSessions before an OpenSession() request.
    /// </para>
    /// <para>
    /// RemoteSessionPool high-performance can be achieved by keeping RPMS server jobs alive even after
    /// secondary sessions are closed.  The pool will maintain MaxSessions - 1 number of jobs alive on the
    /// server.  If the application is finished for awhile needing IdleSessionCount idle jobs, then
    /// TerminateIdleSessions() will release those resources on the server.  New jobs will then be created
    /// on demand.
    /// </para>
    /// 
    /// </summary>
    public interface RemoteSessionPool
    {
        /// <summary>
        /// True if another secondary session can be opened.
        /// </summary>
        /// <remarks>
        /// For this version of BMX, the EHR does not support secondary RemoteSessions and this
        /// method always returns false.  If possible, develop your software using the API so when
        /// secondary RemoteSessions are supported your application will work.
        /// </remarks>
        bool HasAvailableSession { get; }
        
        /// <summary>
        /// The number of additional secondary RemoteSessions that can be opened.
        /// </summary>
        int AvailableSessionCount { get; }

        /// <summary>
        /// The maximum number of RemoteSessions than can be opened
        /// </summary>
        int MaxSessions { get; set; }
      
        /// <summary>
        /// Open a RemoteSession and answer it.  If an idle Session exists, it will be recycled
        /// and answered.  Use Close() on the RemoteSession to release this secondary RemoteSession
        /// back to the pool.
        /// </summary>
        /// <returns>A newly created or recycled RemoteSession or null if the number of RemoteSessions has reach the MaxSessions count</returns>
        RemoteSession OpenSession();

        /// <summary>
        /// Open a new RemoteSession and answer it.  If an idle Session exists, it will be recycled
        /// and answered.  Use Close() on the RemoteSession to release this secondary RemoteSession
        /// back to the pool.  An implementation of Log can be included to trace RPC execution.       
        /// </summary>
        /// <remarks>
        /// Logging is not heavily used in BMXNET40
        /// </remarks>
        /// <returns>A newly created or recycled RemoteSession or null if the number of RemoteSessions has reach the MaxSessions count</returns>
        RemoteSession OpenSession(Log aLog);
   
        //Idle session management
        int IdleSessionCount { get; }        
        void TerminateIdleSessions();
        
        //Lifecycle
        void Close();


    }
}
